//
// The unofficial LEGO Mindstorms RCX SDK
// rcx-lcd.h - control the LCD display
// (c) 1998 by Markus L. Noga <noga@inrialpes.fr>    
//

#ifndef __rcx_lcd_h__
#define __rcx_lcd_h__

///////////////////////////////////////////////////////////////////////////////
//
// Definitions
//
///////////////////////////////////////////////////////////////////////////////

//
// LCD codes
//

typedef enum {
	man_stand	=0x3006,
	man_run		=0x3007,
			
	s1_select	=0x3008,
	s1_active	=0x3009,
			
	s2_select	=0x300A,
	s2_active	=0x300B,
	
	s3_select	=0x300C,
	s3_active	=0x300D,

	a_select	=0x300E,
	a_left		=0x300F,
	a_right		=0x3010,
	
	b_select	=0x3011,
	b_left		=0x3012,
	b_right		=0x3013,
	
	c_select	=0x3014,
	c_left		=0x3015,
	c_right		=0x3016,

	unknown_1	=0x3017, // seemingly without effect. cycle reset?

	// circle and dot codes cycle. cycle state is preserved on powerdown.

	// each dot code should be invoked six times before using the other.
	// mixing them will result in strange behaviour.

	circle		=0x3018, // 0..3 quarters: add one. 4 quarters: reset

	dot		=0x3019, // 0..4 dots: add a dot. 5 dots: reset
				 // dots are added on the right.
	dot_inv		=0x301A, // 0 dots: show 5. 1..4 dots: subtract one
				 // dots are removed on the left

	battery_x	=0x301B,
	
	// the IR display values are mutually exclusive.
	
	ir_half		=0x301C,
	ir_full		=0x301D,
	
	everything	=0x3020
	
} lcd_segment;


///////////////////////////////////////////////////////////////////////////////
//
// Functions
//
///////////////////////////////////////////////////////////////////////////////

//
// note: changes remain invisible until lcd_refresh is invoked.
//

//
// FIXME: register clobbers
//

// show LCD segment
extern inline void lcd_show(lcd_segment segment) {
	register lcd_segment _segment __asm__ ("r6")=segment;
__asm__ __volatile__(
	"jsr @0x1b62:0"
	);
}

// hide LCD segment
extern inline void lcd_hide(lcd_segment segment) {
	register lcd_segment _segment __asm__ ("r6")=segment;
__asm__ __volatile__(
	"jsr @0x1e4a:0"
	);
}

// show signed integer on LCD display (to the left left of the little man)
extern inline void lcd_int(int value) {
__asm__ __volatile__(
		"mov.w %0,r6\n"
		"mov.w #0x3002,r0\n"
		"mov.w r0,@-r7\n"
		"mov.w r6,@-r7\n"
		"mov.w #0x3001,r6\n"
		"jsr @0x1ff2:0\n"
		"adds #0x02,r7\n"
		"adds #0x02,r7\n"
		:			// outputs
		: "r" (value)		// inputs
		: "r0","r6","memory"	// clobbers (final, r0..r5 saved)
		);
}

// show single digit on LCD display (to the right of the little man)
extern inline void lcd_digit(int value) {
__asm__ __volatile__(
		"mov.w %0,r6\n"
		"sub.w r0,r0\n"
		"mov.w r0,@-r7\n"
		"mov.w r6,@-r7\n"
		"mov.w #0x3017,r6\n"
		"jsr @0x1ff2:0\n"
		"adds #0x02,r7\n"
		"adds #0x02,r7\n"
		:			// outputs
		: "r" (value)		// inputs
		: "r0","r6","memory"	// clobbers (final, r0..r5 saved)
		);
}

// clear LCD display
extern inline void lcd_clear(void) {
__asm__ __volatile__(
		"jsr @0x27ac:0"
		:			// outputs
		:			// inputs
		: "r6","memory"		// clobbers (final, r5 saved)
		);
}

// show LCD display contents to the world
extern inline void lcd_refresh(void) {
__asm__ __volatile__(
		"jsr @0x27c8:0"
		:			// outputs
		:			// inputs
		: "r6","memory"		// clobbers (final?)
		);
}

#endif
