//
// The unofficial LEGO Mindstorms RCX SDK
// mm.c - dynamic memory management
// (c) 1998 by Markus L. Noga <noga@inrialpes.fr>    
//

#ifndef	NO_MEMORY_MANAGEMENT

#include "mm-internal.h"
#include "stdlib.h"
#include "tm-internal.h"


///////////////////////////////////////////////////////////////////////////////
//
// Variables
//
///////////////////////////////////////////////////////////////////////////////

size_t *mm_first_free;				// first free block

#ifdef NO_TASK_MANAGEMENT			// we need a non-null
const size_t cpid=0x0001;			// non-0xffff current pid
#endif						// even if not memory managed.

///////////////////////////////////////////////////////////////////////////////
//
// Functions
//
///////////////////////////////////////////////////////////////////////////////

//
// memory block structure:
// 0 1       : pid of owner (0=empty)
// 1 2       : size of data block >> 1
// 3 ... 3+2n: data
//

//
// check for free blocks after this one and join them if possible
// args:   ptr: pointer to size field of current block
// retval: size of block
//
size_t mm_try_join(size_t *ptr) {
	size_t *next=ptr+*ptr+1;
	size_t increase=0;
	
	while(*next==MM_FREE) {
		increase+=*(next+1) + MM_HEADER_SIZE;
		next    +=*(next+1) + MM_HEADER_SIZE;
	}
	return (*ptr)+=increase;
}

//
// update first free block pointer
// arg: pointer to owner field of a memory block to start with.
//
void mm_update_first_free(size_t *start) {
	size_t *ptr=start;
	
	while((*ptr!=MM_FREE) && (ptr>=&mm_start))
		ptr+=*(ptr+1)+MM_HEADER_SIZE;

	mm_first_free=ptr;
}


//
// initialize memory management
//
void mm_init() {
	size_t *current,*next;
	
	current=&mm_start;

	// define memory structure here.
	MM_BLOCK_FREE    (&mm_start);
	MM_BLOCK_RESERVED(0xc000);
	
	// expand last block to encompass all available memory
	*current=(int)(((-(int) current)-2)>>1);
	
	mm_update_first_free(&mm_start);
}


//
// malloc
//
void *malloc(size_t size) {
	size_t *ptr,*next;
	size_t increase;
	
	size=(size+1)>>1;				// only multiples of 2
	ptr=mm_first_free;
	
	while(ptr>=&mm_start) {
		if(*(ptr++)==MM_FREE)  			// free block?
			if(*ptr>=size) {		// big enough?
				*(ptr-1)=(size_t)cpid;	// set owner
							
							// split this block?
				if((*ptr-size)>=MM_SPLIT_THRESH) {
					next=ptr+size+1;
					*(next++)=MM_FREE;
					*(next)=*ptr-size-MM_HEADER_SIZE;
					mm_try_join(next);
					
					*ptr=size;
				}
					
							// was it the first free one?
				if(ptr==mm_first_free+1)
					mm_update_first_free(ptr+*ptr+1);
				
				return (void*) (ptr+1);	
			}
				
			
		ptr+=(*ptr)+1;				// find next block.
	}
	
	return NULL;
}


//
// free. ever heard of free(software paradigm)?
//
void free(void *the_ptr) {
	size_t *ptr=the_ptr,*p2,*next;			// cache it
	
	if(ptr==NULL)
		return;
	
	ptr-=MM_HEADER_SIZE;
	*((size_t*) ptr)=MM_FREE;			// mark as free
	
	p2=&mm_start;
	while(p2!=ptr) {				// we could make free
		next=p2+*(p2+1)+MM_HEADER_SIZE;		// O(1) if we included
		if(*p2=MM_FREE && next==ptr)		// a pointer to the 
			break;				// previous block.
		p2=next;				// I don't want to.
	}
	mm_try_join(p2+1);				// defragment free areas

	if(ptr<mm_first_free || mm_first_free<&mm_start)
		mm_update_first_free(ptr);		// update mm_first_free
}


//
// calloc. have many? or just have cleared?
//
void *calloc(size_t nmemb, size_t size) {
	void *ptr;
	
	size=mul16(size,nmemb);				// FIXME: overflows?
	
	if((ptr=malloc(size))!=NULL)
		memset(ptr,0,size);
		
	return ptr;
}

//
// free all blocks allocated by the current process.
//
void mm_reaper() {
	size_t *ptr,*next;
	
	// pass 1: mark as free	
	ptr=&mm_start;
	while(*ptr>=(size_t) &mm_start) {
		if(*ptr==(size_t)cpid)
			*ptr=MM_FREE;
		ptr+=*(ptr+1)+MM_HEADER_SIZE;
	}
	
	// pass 2: defragment free areas
	ptr=&mm_start;
	while(*ptr>=(size_t) &mm_start) {
		if(*(ptr++)==MM_FREE)
			mm_try_join(ptr);
		ptr+=*ptr+1;
	}
}	
	
#endif
