//
// The unofficial LEGO Mindstorms RCX SDK
// tm.c - task management
// (c) 1998 by Markus L. Noga <noga@inrialpes.fr>    
//

#ifndef NO_TASK_MANAGEMENT

#include "tm-internal.h"
#include "bitops.h"
#include "stdlib.h"
#include "rcx-lcd.h"


///////////////////////////////////////////////////////////////////////////////
//
// Variables
//
///////////////////////////////////////////////////////////////////////////////

process_data pd_single;				// single process process data

process_data *cpid;				// ptr to current process data



///////////////////////////////////////////////////////////////////////////////
//
// Functions
//
///////////////////////////////////////////////////////////////////////////////

//
// init task management
// called in single tasking mode before task setup.
//
void tm_init(void) {
	pd_single.next=NULL;
	pd_single.pstate=P_RUNNING;
	cpid=&pd_single;		
}

//
// start task management 
// called in single tasking mode after task setup
//
void tm_start(void) {
	unsigned b;

	if(pd_single.next==NULL)		// no tasks?
		return;	

	__asm__ __volatile__("orc #0x80,ccr\n");// no IRQs, please
						// tasks will reactivate them
	
	*((unsigned char*)T1_CR)  =0x00;	// timer 1 off
	*((unsigned char*)T1_CNT) =0x00;	// counter reset
	*((unsigned char*)T1_CORA)=0xf0;	// set const A
	*((void**)T1_IRQA)=&tm_switcher;	// set IRQ handler
	
	// set clock=processor/2048 (STCR bit 0x1 / T1_CR mask 0x03)
	// => 7800 ticks / second, timeslice = const a * 0,128 ms
	// set IRQ a enable (T1_CR mask 0x40)
	// enable counter reset on IRQ A (T1_CR mask 0x08)
	bit_clear(STCR,0x1);		
	*((unsigned char*) T1_CR)=0x4b; 	// go!

	__asm__ __volatile__(
			"jsr _tm_switcher\n"	// we will be back...
			"andc #0x7f,ccr\n"	// re-activate IRQs
			:::"r6","cc","memory"
			);
}

//
// shutdown task management
// called by scheduler in multitasking mode
//
void tm_stop(void) {
	*((unsigned char*) T1_CR)=0x00;		// irq & counting off
	*((unsigned*)T1_IRQA)    =0x046a;	// empty handler
}


//
// the process scheduler
// arg   : current task's current stack pointer
// retval: new task's current stack pointer
//
// actual context switches performed by tm_switcher (assembler wrapper)
//
size_t *tm_scheduler(size_t *old_sp) {
	process_data *next;			// next process to execute

#ifndef	NO_RUNNING_MAN	
	static unsigned sequence=0;

	sequence++;				// yeah we're running
	if(sequence==10) {
		lcd_show(man_stand);
		lcd_refresh();
	} else if(sequence==20) {
		lcd_show(man_run);
		lcd_refresh();
		sequence=0;
	}
#endif	// NO_RUNNING_MAN
	
		
	// determine new process. right now, just take the next one.
	// this can be any type of scheduler.
	
	next=cpid->next;

	if(cpid->pstate==P_ZOMBIE) {	// did this one finish?
		cpid->prev->next = cpid->next;	// remove from
		cpid->next->prev = cpid->prev;	// queue
		
		//
		// FIXME: we're on that stack frame being freed right now!
		//
		free(cpid->stack_base);	// free stack
		free(cpid);		// free process data
		
		//
		// FIXME: exit code?
		//
	
		if(next==cpid) {			// last processes dead?
			tm_stop();			// stop scheduler IRQ
			pd_single.next=NULL;		// no children
			next=&pd_single;		// go single tasking.
		}

	} else {
		cpid->sp_save=old_sp;	// save sp
		cpid->pstate=P_SLEEPING;
	}
	
	cpid=next;			// execute next process.
	cpid->pstate=P_RUNNING;

	return cpid->sp_save;
}


//
// execute a memory image.
// args: start address of code to execute
//       stack size for new process.
// retval: -1: fail, else pid
//
pid_t execi(int (*code_start)(void),size_t stack_size) {
	process_data *pd;
	size_t *sp;

	// get memory
	//
	// task & stack area belong to parent process
	// they aren't freed by mm_reaper()
	//
	if((pd=malloc(sizeof(process_data)))==NULL)	
		return -1;

	if((sp=malloc(stack_size))==NULL) {
		free(pd);
		return -1;
	}

	pd->stack_base=sp;		// these we know already.
	
	sp+=(stack_size>>1);		// setup initial stack
				
	// when main() returns a value, it passes it in r0
	// as r0 is also the register to pass single int arguments by
	// gcc convention, we can just put the address of exit on the stack.
	
	*(--sp)=(size_t) &exit;
	
		
	// we have to construct a stack stub so tm_switcher and the
	// ROM routine can fill in all the right values on startup.
	
	*(--sp)=(size_t) code_start;	// entry point    < these two are for
	*(--sp)=0;			// ccr            < rte in ROM
	*(--sp)=0;			// r6                    < pop r6 in ROM
	*(--sp)=0x051c;			// our ROM return point  < rts in tm_switcher
	*(--sp)=0;			// r0..r5	  < pop r0..r5 in tm_switcher
	*(--sp)=0;
	*(--sp)=0;
	*(--sp)=0;
	*(--sp)=0;
	*(--sp)=0;
	
	pd->sp_save=sp;			// save sp for tm_scheduler
	pd->pstate=P_SLEEPING;		// task is waiting for execution

	pd->parent=cpid;		// insert into queue
	
	if(cpid==&pd_single && cpid->next==NULL)
		cpid->next=pd->prev=pd->next=pd;	// start queue
	else {		
		pd->next=cpid->next;	// avoid pd_single problems
		pd->prev=cpid->next->prev;	
		pd->prev->next=pd;
		pd->next->prev=pd;
	}
			
	return (pid_t) pd;		// pid = (pid_t) &process_data_struct
}


//
// exit task, returning code
// FIXME: for now, scrap the code.
//
void exit(int code) {
	mm_reaper();				// free dynamic memory
	cpid->pstate=P_ZOMBIE;			// come, task reaper!

	while(1)
		;
}

#endif
