//
// The unofficial LEGO Mindstorms RCX SDK
// direct-ir.c - direct IR port access
// (c) 1998 by Markus L. Noga <noga@inrialpes.fr>    
//

#ifndef NO_DIRECT_IR

#include "direct-ir.h"
#include "h8.h"
#include "rcx-irq.h"

#include "conio.h"
#include "rcx-lcd.h"


#define RX_BUF_SIZE	256
#define TX_BUF_SIZE	256

///////////////////////////////////////////////////////////////////////////////
//
// Variables
//
///////////////////////////////////////////////////////////////////////////////

unsigned char dir_rx_buf[RX_BUF_SIZE];		// rx buffer
unsigned char *dir_rx_end;			// rx buffer end (last+1)
volatile unsigned char *dir_rx_write;		// ptr to next byte to write in rx irq
unsigned char *dir_rx_read;			// ptr to next byte to read

#define RX_OK		0
#define RX_ERROR	-1

volatile int dir_rx_state;			// rx error flag


unsigned char *dir_tx_end;			// tx buffer end (last+1)
volatile unsigned char *dir_tx_read;		// ptr to next byte to be sent.

unsigned char *dir_tx_verify;			// ptr to compare sent bytes
						// with received ones.
						// null: don't compare.

#define TX_ACTIVE	0			// transmission states
#define TX_OK		1
#define TX_MISMATCH	-1
	
volatile int dir_tx_state;			// transmission state

///////////////////////////////////////////////////////////////////////////////
//
// Functions
//
///////////////////////////////////////////////////////////////////////////////

//
// C core for the rx byte received interrupt
//
void dir_rx_c(void) {
	unsigned char received=S_RDR;
	unsigned char *bufptr;		

	if(dir_tx_verify) {
		// verify sent data by checking if we receive it, too
		//
		if(received!=*(dir_tx_verify++)) {
			// quit sending on mismatch
			//
			S_CR&=~(SCR_TRANSMIT | SCR_TX_IRQ | SCR_TE_IRQ);
			dir_tx_state=TX_MISMATCH;
			dir_tx_verify=NULL;
		}
		if(dir_tx_verify==dir_tx_end) {
			// everything sent -> OK
			//
			dir_tx_state=TX_OK;
			dir_tx_verify=NULL;
		}
	} else {		
		// receive data
		//
		bufptr=(unsigned char*) dir_rx_write;
		
		*bufptr=received;
		if((++bufptr)>=dir_rx_end)
			bufptr=dir_rx_buf;
		
		// FIXME: catch buffer overflow
				
		dir_rx_write=bufptr;
	}
}


//
// assembler wrapper for the rx byte received interrupt
//
void dir_rx_handler(void);

__asm__("
_dir_rx_handler:
		; r6 saved by ROM
		
		mov.w	r0,@-r7			; save all registers
		mov.w	r1,@-r7	
		mov.w	r2,@-r7		
		mov.w	r3,@-r7		
		mov.w	r4,@-r7		
		mov.w	r5,@-r7		
		
		jsr	_dir_rx_c		; call scheduler
		
		bclr	#0x6,@0xdc:8		; clear receive flag in SSR
		
		mov.w	@r7+,r5			; restore all registers
		mov.w	@r7+,r4
		mov.w	@r7+,r3
		mov.w	@r7+,r2
		mov.w	@r7+,r1
		mov.w	@r7+,r0
		
		; r6 will be restored by ROM

		rts				; return 
	");

//
// C core for the rx error interrupt
//
void dir_rxerror_c(void) {
	if(dir_tx_verify) {
		// quit sending on mismatch
		//
		S_CR&=~(SCR_TRANSMIT | SCR_TX_IRQ | SCR_TE_IRQ);
		dir_tx_state=TX_MISMATCH;
		dir_tx_verify=NULL;
	} else
		dir_rx_state=RX_ERROR;
}


//
// assembler wrapper for the rx error interrupt
//
void dir_rxerror_handler(void);

__asm__("
_dir_rxerror_handler:
		; r6 saved by ROM
		
		mov.w	r0,@-r7			; save all registers
		mov.w	r1,@-r7	
		mov.w	r2,@-r7		
		mov.w	r3,@-r7		
		mov.w	r4,@-r7		
		mov.w	r5,@-r7		
		
		jsr	_dir_rxerror_c		; call scheduler
		
		mov.b	@0xdc:8,r0l
		and.b	#0xc7,r0l
		mov.b	r0l,@0xdc:8		; clear error flags in SSR
		
		mov.w	@r7+,r5			; restore all registers
		mov.w	@r7+,r4
		mov.w	@r7+,r3
		mov.w	@r7+,r2
		mov.w	@r7+,r1
		mov.w	@r7+,r0
		
		; r6 will be restored by ROM

		rts				; return 
	");

//
// assembler tx byte sent interrupt
// write next byte if there's one left, otherwise unhook irq.
//
void dir_tx_handler(void);

__asm__("
_dir_tx_handler:
		; r6 saved by ROM
		
		mov.w	r0,@-r7
		
		mov.w @_dir_tx_read,r0		; dir_tx_read -> r0
		mov.w @_dir_tx_end,r6

		cmp.w r0,r6			; transmission finished?
		beq  0f
		
		  mov.b @r0+,r6l		; *(ptr++) -> r6
		  mov.b r6l,@0xdb:8		; r6 -> transmit data register
		  
		  bclr	#0x7,@0xdc:8		; clear transmit buffer empty flag
		  
		  bra 1f
		
		0:bclr  #0x7,@0xda:8		; disable transmission irq
		
	      1:mov.w r0,@_dir_tx_read		; store new pointer
	      
	        mov.w @r7+,r0
		rts
	");		

//
// assembler tx end interrupt
// shutdown transmission
//
void dir_txend_handler(void);

__asm__("
_dir_txend_handler:
		; r6 saved by ROM

		mov.b	@0xda:8,r6l		; shutdown transmit & irqs
		and.b	#0x5b,r6l
		mov.b	r6l,@0xda:8
		
		mov.b	@0xdc:8,r6l		; clear transmit status flags
		and.b	#0x7b,r6l
		mov.b	r6l,@0xdc:8
		
		rts
	");
			
//
// initialize IR port
//
void dir_init(void) {
	S_CR=SCR_INT_CLOCK;			// disable everything

	// set up IR carrier for transmission
	//
	__asm__("mov.b	#0x9,r6l		; timer 1 control register
	         mov.b	r6l,@0xd0:8		; = clear on match a, p/8 clock
	         mov.b	#0x13,r6l		; timer 1 control status register
	         mov.b	r6l,@0xd1:8		; = output toggle on match a
				  
                 mov.b	#0x1a,r6l		; timer 1 const a = 0x1a
                 mov.b	r6l,@0xd2:8

		 ");
		 
	// *((unsigned char*)0xffb7)&=~0x01;	// set IR xmit range 
	dir_tx_verify=NULL;			// init transmit verification
	dir_tx_state =TX_OK;			// data
		
	dir_rx_write=dir_rx_buf;
	dir_rx_read =dir_rx_buf;
	dir_rx_end  =dir_rx_buf+RX_BUF_SIZE;	// init receive buffer pointers
	
	// 8/O/1, 2400 baud is RCX standard
	// 4800 baud seems to work in close range.
	S_MR =SMR_ASYNC | SMR_8BIT | SMR_P_ODD | SMR_1STOP | SMR_CLOCK;
	S_BRR=BAUD_2400;

	RX_IRQ      =&dir_rx_handler;		// set irq handlers
	RX_ERROR_IRQ=&dir_rxerror_handler;
	TX_IRQ	    =&dir_tx_handler;
	TX_END_IRQ  =&dir_txend_handler;

	S_SR&=~(SSR_RECV_FULL || SSR_OVERRUN_ERR
	   || SSR_FRAMING_ERR || SSR_PARITY_ERR);
						// clear recv flags in SSR
	S_CR|=SCR_RECEIVE | SCR_RX_IRQ;		// enable receiving and irqs
}


//
// shutdown IR port
//
void dir_shutdown(void) {
	S_CR=0x00;				// everything off
	
	__asm__("mov.b	#0x0,r6l		; timer 1 control register
	         mov.b	r6l,@0xd0:8		; = all off
		");	
}


//
// write to IR port, blocking.
// return code: nr of bytes written. -1 -> error.
//
size_t dir_write(void* const buf,size_t len) {
	if(buf==NULL || len==0)			// catch empty cases
		return 0;
	
 	dir_tx_read=buf;			// what to transmit
	dir_tx_end=buf+len;			
	
	dir_tx_verify=buf;			// checks and balances
	dir_tx_state=TX_ACTIVE;
	
	S_SR&=~(SSR_TRANS_EMPTY | SSR_TRANS_END);	// clear flags
	S_CR|=SCR_TRANSMIT | SCR_TX_IRQ | SCR_TE_IRQ;	// enable transmit & irqs
	
	while(dir_tx_state==TX_ACTIVE)	
		;				// FIXME: scheduler integration

	if(dir_tx_state==TX_OK)
		return len;			// successful xmit
	
	return -1;				// default is error
}


//
// read from IR port, blocking.
// return code: nr of bytes written. -1 -> error.
//
size_t dir_read(void* buf,size_t len) {
	unsigned char* ptr;
	
	if(buf==NULL || len==0)			// catch empty cases
		return 0;

	ptr=buf;
	buf+=len;
	dir_rx_state=RX_OK;
		
	for(; ptr<(unsigned char*)buf; ) {
		// FIXME: scheduler integration
		//
		while(dir_rx_read==dir_rx_write)
			if(dir_rx_state!=RX_OK)
				goto dir_rx_error_handler;
		
		*(ptr++)=*(dir_rx_read++);	// copy to target buffer
		if(dir_rx_read==dir_rx_end)
			dir_rx_read=dir_rx_buf;
	}
	return len;				// successful read
		
  	dir_rx_error_handler:
	return -1;				// default is error
}

#endif
