//
// The unofficial LEGO Mindstorms RCX SDK
// rcx-lcd.h - control the LCD display
// (c) 1998 by Markus L. Noga <noga@inrialpes.fr>    
//

#ifndef __rcx_lcd_h__
#define __rcx_lcd_h__

///////////////////////////////////////////////////////////////////////////////
//
// Definitions
//
///////////////////////////////////////////////////////////////////////////////

//
// LCD segment codes
//

typedef enum {
	man_stand	=0x3006,
	man_run		=0x3007,
			
	s1_select	=0x3008,
	s1_active	=0x3009,
			
	s2_select	=0x300A,
	s2_active	=0x300B,
	
	s3_select	=0x300C,
	s3_active	=0x300D,

	a_select	=0x300E,
	a_left		=0x300F,
	a_right		=0x3010,
	
	b_select	=0x3011,
	b_left		=0x3012,
	b_right		=0x3013,
	
	c_select	=0x3014,
	c_left		=0x3015,
	c_right		=0x3016,

	unknown_1	=0x3017, // seemingly without effect. cycle reset?

	// circle and dot codes cycle. cycle state is preserved on powerdown.

	// each dot code should be invoked six times before using the other.
	// mixing them will result in strange behaviour.

	circle		=0x3018, // 0..3 quarters: add one. 4 quarters: reset

	dot		=0x3019, // 0..4 dots: add a dot. 5 dots: reset
				 // dots are added on the right.
	dot_inv		=0x301A, // 0 dots: show 5. 1..4 dots: subtract one
				 // dots are removed on the left

	battery_x	=0x301B,
	
	// the IR display values are mutually exclusive.
	
	ir_half		=0x301C,
	ir_full		=0x301D,
	
	everything	=0x3020
	
} lcd_segment;


//
// LCD number display styles
//
// note: signed and unsigned are taken by the C programming language
//

typedef enum {
	digit		=0x3017,	// single digit on the right
					// works only with comma_style digit
			
	sign		=0x3001,	// signed, no leading zeros
	unsign		=0x301F		// unsigned, 0 displayed as 0000
			
} lcd_number_style;


//
// LCD comma display styles
//

typedef enum {
	digit_comma	=0x0000,	// single digit on the right
					// works only with number_style digit
			
	e0		=0x3002,	// whole
	e_1		=0x3003,	// 10ths
	e_2		=0x3004,	// 100ths
	e_3		=0x3005,	// 1000ths, problematic with negatives
			
} lcd_comma_style;			// lcd display comma style

	
//
// Convenient macros
//	

#define lcd_clock(t)	lcd_number(t,unsign,e_2)
#define lcd_int(i)	lcd_number(i,sign,e0)
#define lcd_unsigned(u)	lcd_number(u,unsign,e0)
#define lcd_digit(d)	lcd_number(d,digit,digit_comma)


///////////////////////////////////////////////////////////////////////////////
//
// Functions
//
///////////////////////////////////////////////////////////////////////////////

//
// note: changes remain invisible until lcd_refresh is invoked.
//

// show LCD segment
extern inline void lcd_show(lcd_segment segment) {
	register lcd_segment _segment __asm__ ("r6")=segment;
__asm__ __volatile__(
	"jsr @0x1b62:0\n"
	:				// outputs
	: 			 	// inputs
	: "r6","cc","memory"		// clobbers (r4,r5 saved -> final.)
	);
}

// hide LCD segment
extern inline void lcd_hide(lcd_segment segment) {
	register lcd_segment _segment __asm__ ("r6")=segment;
__asm__ __volatile__(
	"jsr @0x1e4a:0\n"
	:				// outputs
	: 			 	// inputs
	: "r6","cc","memory"		// clobbers  (final)
	);
}

// show number on LCD display
// number and comma styles see above.
extern inline void lcd_number(int i,lcd_number_style n,
				    lcd_comma_style c  ) {
__asm__ __volatile__(
	"mov.w %2,@-r7\n"
	"mov.w %0,@-r7\n"
	"mov.w %1,r6\n"
	"jsr @0x1ff2:0\n"
	"adds #0x02,r7\n"
	"adds #0x02,r7\n"
	:				// outputs
	: "r" (i), "r" (n), "r" (c) 	// inputs
	: "r6","cc"			// clobbers (final)
	);
}

// clear LCD display
extern inline void lcd_clear(void) {
__asm__ __volatile__(
	"jsr @0x27ac:0\n"
	:				// outputs
	:				// inputs
	: "r6","cc","memory"		// clobbers (final, r5 saved)
	);
}

/*// show LCD display contents to the world
#define lcd_refresh() \
__asm__ __volatile__("jsr @0x27c8:0":::"r6","cc","memory")

*/
// show LCD display contents to the world
extern inline void lcd_refresh(void) {
__asm__ __volatile__(		
	"jsr @0x27c8:0"
	:				// outputs
	:				// inputs
	: "r6","cc","memory"		// clobbers (final, r3..r5 saved)
	);
}


#endif
